<?php


if (!class_exists('AopsCurlHelper')) {
	class AopsCurlHelper extends Aops {

		
		// errors source: https://curl.haxx.se/libcurl/c/libcurl-errors.html 
		public static $sslErrors = array(
			CURLE_SSL_CONNECT_ERROR			=>  'A problem occurred somewhere in the SSL/TLS handshake. Could be certificates issue (file formats, paths, permissions), passwords, and others.',
			51 /* CURLE_PEER_FAILED_VERIFICATION */	=> 'The remote server\'s SSL certificate is not OK.',
			CURLE_SSL_CERTPROBLEM			=> 'Problem with the local client certificate',
			CURLE_SSL_CACERT				=> 'Peer certificate cannot be authenticated with known CA certificates.',
			77 /* CURLE_SSL_CACERT_BADFILE */	=> 'Problem with reading the SSL CA cert (possible wrong path ot access rights)',
			91 /*CURLE_SSL_INVALIDCERTSTATUS */ => 'Sertificate\'s status verification failes',
			60								=> 'Unable to get local issuer certificate'
		);

			
			
		/** 
		 * Send a POST requst using cURL 
		 * @param string $url to request 
		 * @param array $post values to send 
		 * @param array $options for cURL 
		 * @return string 
		*/ 
		public static function post($url, array $post = NULL, array $options = array()) {
			if (!function_exists('curl_init')) {
				self::saveSslStatus( array( 'curl' => false, 'result'	=> false ) );
				Aops::log("AopsCurlHelper::post() --- curl_init not found");
				return false;
			}
			
			$defaults = array( 
				CURLOPT_POST => 1, 
				CURLOPT_HEADER => 0, 
				CURLOPT_URL => $url, 
				CURLOPT_FRESH_CONNECT => 1, 
				CURLOPT_RETURNTRANSFER => 1, 
				CURLOPT_FORBID_REUSE => 1, 
				CURLOPT_TIMEOUT => 8, 
				CURLOPT_POSTFIELDS => http_build_query($post) 
			); 

			$ch = curl_init(); 
			curl_setopt_array($ch, ($options + $defaults));

			if( ! $result = curl_exec($ch)) { 
				$errorData = array(
					'action'		=> 'post', 
					'url'			=> $url, 
					'post'			=> $post, 
					'options'		=> $options,
					'error_number'	=> curl_errno($ch),
					'error_message'	=> curl_error($ch),
				);
				curl_close($ch); 
				$result = self::processCurlError($errorData);
			}
			else {
				self::saveSslStatus(array(
					'local_ssl' => true,
					'ssl_location' => true,
					'result'	=> true
				));
				curl_close($ch); 
			}

			self::saveSslStatus('curl', true);
			return $result; 
		} 

		/** 
		 * Send a GET requst using cURL 
		 * @param string $url to request 
		 * @param array $get values to send 
		 * @param array $options for cURL 
		 * @return string 
		*/ 
		public static function get($url, array $get = NULL, array $options = array()) {
			if (!function_exists('curl_init')) {
				self::saveSslStatus('curl', false);
				Aops::log("AopsCurlHelper::get() --- curl_init not found, using wp_safe_remote_get instead");
				return wp_safe_remote_get($url,$get);
			}
			
			$defaults = array( 
				CURLOPT_URL => $url. (strpos($url, '?') === FALSE ? '?' : ''). http_build_query($get), 
				CURLOPT_HEADER => 0, 
				CURLOPT_RETURNTRANSFER => TRUE, 
				CURLOPT_TIMEOUT => 8 
			); 

			$ch = curl_init(); 
			curl_setopt_array($ch, ($options + $defaults)); 
			if( ! $result = curl_exec($ch)) { 
				$errorData = array(
					'action'		=> 'get', 
					'url'			=> $url,
					'get'			=> $get,
					'options'		=> $options,
					'error_number'	=> curl_errno($ch),
					'error_message'	=> curl_error($ch),
				);
				curl_close($ch); 
				$result = self::processCurlError($errorData);
			}
			else {
				self::saveSslStatus(array(
					'local_ssl' => true,
					'ssl_location' => true,
					'result'	=> true
				));
				curl_close($ch); 
			}
			
			self::saveSslStatus(array('curl' => true));
			return $result; 
		} 

		/** 
		 * Send a POST requst using Basic HTTP Autentification via cURL 
		 * @param string $url to request 
		 * @param string $username  
		 * @param string $password 
		 * @param array $post values to send 
		 * @param array $options for cURL 
		 * @return string 
		*/ 
		public static function post_auth($url, $username, $password, array $post = NULL, array $options = array()) {
			if (!function_exists('curl_init')) {
				self::saveSslStatus( array( 'curl' => false, 'result'	=> false ) );
				Aops::log("AopsCurlHelper::post_auth() --- curl_init not found");
				return false;
			}
			
			Aops::log("AopsCurlHelper::post_auth() $username, $password");
			Aops::log($post);
			$defaults = array( 
				CURLOPT_POST => 1, 
				CURLOPT_HEADER => 0, 
				CURLOPT_URL => $url, 
				CURLOPT_FRESH_CONNECT => 1, 
				CURLOPT_RETURNTRANSFER => 1, 
				CURLOPT_FORBID_REUSE => 1, 
				CURLOPT_TIMEOUT => 8, 
				CURLOPT_POSTFIELDS => http_build_query($post) 
			); 
			
			$ch = curl_init(); 
			curl_setopt_array($ch, ($options + $defaults));

			curl_setopt($ch, CURLOPT_USERPWD, $username . ":" . $password); 

			if( ! $result = curl_exec($ch)) {
				$errorData = array(
					'action'		=> 'post_auth', 
					'url'			=> $url, 
					'username'		=> $username, 
					'password'		=> $password, 
					'post'			=> $post, 
					'options'		=> $options,
					'error_number'	=> curl_errno($ch),
					'error_message'	=> curl_error($ch),
				);
				curl_close($ch); 
				$result = self::processCurlError($errorData);
			}
			else {
				self::saveSslStatus(array(
					'local_ssl' => true,
					'ssl_location' => true,
					'result'	=> true
				));
				curl_close($ch); 
			}
			
			self::saveSslStatus(array('curl' => true));
			return $result; 
		} 

		public static function post_auth_json($url, $username, $password, array $post = NULL, array $options = array()) {
			if (!function_exists('curl_init')) {
				self::saveSslStatus('curl', false);
				Aops::log("AopsCurlHelper::post_auth_json() --- curl_init not found");
				return false;
			}
			
			Aops::log("AopsCurlHelper::post_auth_json() $username, $password");
			Aops::log($post);
			$defaults = array( 
				CURLOPT_POST => 1, 
				CURLOPT_HEADER => 'Content-Type: application/json',
				CURLOPT_URL => $url, 
				CURLOPT_FRESH_CONNECT => 1, 
				CURLOPT_RETURNTRANSFER => 1, 
				CURLOPT_FORBID_REUSE => 1, 
				CURLOPT_TIMEOUT => 8, 
				CURLOPT_POSTFIELDS => json_encode($post) 
			); 

			$ch = curl_init(); 
			curl_setopt_array($ch, ($options + $defaults));

			curl_setopt($ch, CURLOPT_USERPWD, $username . ":" . $password); 

			if( ! $result = curl_exec($ch)) {
				$errorData = array(
					'action'		=> 'post_auth_json', 
					'url'			=> $url, 
					'username'		=> $username, 
					'password'		=> $password, 
					'post'			=> $post, 
					'options'		=> $options,
					'error_number'	=> curl_errno($ch),
					'error_message'	=> curl_error($ch),
				);
				curl_close($ch); 
				$result = self::processCurlError($errorData);
			}
			else {
				self::saveSslStatus(array(
					'local_ssl' => true,
					'ssl_location' => true,
					'result'	=> true
				));
				curl_close($ch); 
			}
			
			self::saveSslStatus(array('curl' => true));
			return $result; 
		} 

		/** 
		 * Send a GET requst using cURL 
		 * @param string $url to request 
		 * @param string $username  
		 * @param string $password 
		 * @param array $get values to send 
		 * @param array $options for cURL 
		 * @return string 
		*/ 
		public static function get_auth($url, $username, $password, array $get = NULL, array $options = array()) {
			
			if (!function_exists('curl_init')) {
				self::saveSslStatus( array( 'curl' => false, 'result'	=> false ) );
				Aops::log("AopsCurlHelper::get_auth() --- curl_init not found");
				return false;
			}
			
			
			$defaults = array( 
				CURLOPT_URL => $url. (strpos($url, '?') === FALSE ? '?' : ''). http_build_query($get), 
				CURLOPT_HEADER => 0, 
				CURLOPT_RETURNTRANSFER => TRUE, 
				CURLOPT_TIMEOUT => 8
			); 
		
			$ch = curl_init(); 
			curl_setopt_array($ch, ($options + $defaults)); 
			curl_setopt($ch, CURLOPT_USERPWD, $username . ":" . $password); 


			if( ! $result = curl_exec($ch)) {
				$errorData = array(
					'action'		=> 'get_auth', 
					'url'			=> $url, 
					'username'		=> $username, 
					'password'		=> $password, 
					'get'			=> $get,
					'options'		=> $options,
					'error_number'	=> curl_errno($ch),
					'error_message'	=> curl_error($ch),
				);
				curl_close($ch); 
				$result = self::processCurlError($errorData);
			}
			else {
				self::saveSslStatus(array(
					'local_ssl' => true,
					'ssl_location' => true,
					'result'	=> true
				));
				curl_close($ch); 
			}
			
			self::saveSslStatus(array('curl' => true));
			
			return $result; 
		}
		
		/**
		 * 
		 * $errorData = array(
					'action'		=> get_auth |
					'url'			=> $url, 
					'username'		=> $username, 
					'password'		=> $password, 
					'get'			=> $get, 
					'options'		=> $options,
					'error_number'	=> curl_errno($ch),
					'error_message'	=> curl_error($ch),
				);
		 * @param array $errorData
		 */
		public static function processCurlError($errorData) {
			if (self::checkIfCurlErrorSslRelated($errorData['error_number'], $errorData['error_message'])) {
				
				$message =  "Curl error #" . $errorData['error_number'] . ' : ' . $errorData['error_message'] . '.';
				
				$curlErrorNumber = $errorData['error_number'];
				if (isset(self::$sslErrors[$curlErrorNumber])) {
					$message .=  self::$sslErrors[$curlErrorNumber];
				}

				self::saveSslStatus(array(
					'local_ssl' => !in_array($curlErrorNumber, array(60,CURLE_SSL_CERTPROBLEM)),
					'ssl_location' => !in_array($curlErrorNumber, array(77)),
					'curl'		=> true,
					'result'	=> false,
					'message'	=> $message
				));
				
				if (self::checkIfSslDisabled()) {
					$result = self::repeatRequestWithoutSsl($errorData);

					Aops::log("AopsCurlHelper:: " . $errorData['action'] . " (" . $errorData['url'] . ', ' . $errorData['username'] 
						. ") : using insecure connection due to SSL misconfiguration. $message )");

				}
				else {
					$result = false;
					Aops::log("AopsCurlHelper:: " . $errorData['action'] . " : Failed to get data from remote server.  $message ");
				}
			}
			
			return $result;
		}
		
		public static function checkIfCurlErrorSslRelated($curlErrorNumber, $curlErrorMessage) {
	
			if (isset(self::$sslErrors[$curlErrorNumber])) {
				Aops::log('AopsCurlHelper : SSL problem detected. Description: ' . self::$sslErrors[$curlErrorNumber]);
				return true;
			}
			return false;
		}
		
		public static function checkIfSslIsOn($options) {
			// by default SSL is on, so if there are no 'CURLOPT_SSL_VERIFYPEER' value specified, consider SSL to be enabled
			return (array_key_exists(CURLOPT_SSL_VERIFYPEER, $options) ? $options[CURLOPT_SSL_VERIFYPEER] : true);
		}
		
		public static function repeatRequestWithoutSsl($errorData) {
				//$requestType, $url, $username, $password, array $data = NULL, array $options = array()) {
			$options = $errorData['options'] + array(CURLOPT_SSL_VERIFYPEER =>  false);
			$url = $errorData['url'];
			$username = isset ($errorData['username']) ? $errorData['username'] : '';
			$password = isset ($errorData['password']) ? $errorData['password'] : '';
			
			$result = false;
			switch ($errorData['action']):
				case 'get': $result =					self::get($url, $errorData['get'], $options); break;
				case 'post': $result =					self::post($url, $errorData['post'], $options); break;
				case 'get_auth': $result =				self::get_auth($url, $username, $password, $errorData['get'], $options); break;
				case 'post_auth': $result =				self::post_auth($url, $username, $password, $errorData['post'], $options); break;
				case 'post_auth_json': $result =		self::post_auth_json($url, $username, $password, $errorData['post'], $options); break;
				default: return false;
			endswitch;
					
			return $result;
		}
		
		/**
		* Saves plugin 'ssl_connection_status' option
		*/
		public static function saveSslStatus($status) {
			$optionValue  = get_option(self::$prefix . 'ssl_connection_status');

			if (is_array($optionValue)) {
				foreach ($status as $statusSetting => $value) {
					$optionValue[$statusSetting] = $value;
				}
			}
			else {
				$optionValue = $status;
			}
			
			//Aops::log($optionValue);
			update_option(self::$prefix . 'ssl_connection_status', $optionValue);
		}

}
	
	
}